#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <algorithm>
#include <string>

using namespace std;

//----------------------------------------------------------
// 1) Osnovno: preopterećenje operatora za prost tip (Tacka)
//    - operator<< za ispis
//    - operator== i operator< za poređenje
//----------------------------------------------------------
struct Tacka {
    int x;
    int y;
};

// ispis u obliku (x,y)
ostream& operator<<(ostream& os, const Tacka& t) {
    return os << "(" << t.x << "," << t.y << ")";
}

// jednakost: oba koord. jednaka
bool operator==(const Tacka& a, const Tacka& b) {
    return a.x == b.x && a.y == b.y;
}

// uređenje po x pa po y
bool operator<(const Tacka& a, const Tacka& b) {
    if (a.x != b.x) return a.x < b.x;
    return a.y < b.y;
}

void demo_operators_basic() {
    cout << "\n[demo_operators_basic]\n";

    Tacka p{1,2}, q{1,2}, r{0,5};

    cout << "p = " << p << ", q = " << q << ", r = " << r << "\n";
    cout << "p == q ? " << (p == q) << "\n";
    cout << "p < r ?  " << (p < r) << "\n";
    cout << "r < p ?  " << (r < p) << "\n";
}

//----------------------------------------------------------
// 2) Korišćenje operator< u std::set
//    - ključna motivacija za preopterećenje <
//----------------------------------------------------------
void demo_set_of_points() {
    cout << "\n[demo_set_of_points]\n";

    set<Tacka> S;    // koristi operator< za uređenje

    S.insert({1,2});
    S.insert({0,0});
    S.insert({1,2}); // duplikat, neće se dodati
    S.insert({-1,5});

    cout << "Sadrzaj skupa:\n";
    for (const auto& t : S) {
        cout << "  " << t << "\n";
    }

    // pretraga u skupu
    Tacka trazena{1,2};
    auto it = S.find(trazena);
    cout << "Trazenje tacke " << trazena << ": "
         << (it != S.end() ? "nadjena\n" : "nije nadjena\n");
}

//----------------------------------------------------------
// 3) Preopterećenje operator+ za jednostavan tip (Tacka)
//    - (x1+x2, y1+y2)
//    - nije nužno, ali lepo ilustruje ideju
//----------------------------------------------------------
Tacka operator+(const Tacka& a, const Tacka& b) {
    return {a.x + b.x, a.y + b.y};
}

void demo_operator_plus() {
    cout << "\n[demo_operator_plus]\n";

    Tacka a{1,2}, b{3,4};
    Tacka c = a + b;

    cout << a << " + " << b << " = " << c << "\n";
}

//----------------------------------------------------------
// 4) Funktor kao komparator za set/map
//    - Tacka po y opadajuće, pa po x rastuće
//----------------------------------------------------------
struct TackaPoYOpadajuce {
    bool operator()(const Tacka& a, const Tacka& b) const {
        if (a.y != b.y) return a.y > b.y;   // veći y "pre" (opadajuće)
        return a.x < b.x;                   // kod istog y, manji x "pre"
    }
};

void demo_functor_as_comparator() {
    cout << "\n[demo_functor_as_comparator]\n";

    set<Tacka, TackaPoYOpadajuce> S;

    S.insert({0,0});
    S.insert({1,5});
    S.insert({2,5});
    S.insert({3,1});

    cout << "Set sa TackaPoYOpadajuce:\n";
    for (const auto& t : S) {
        cout << "  " << t << "\n";
    }
}

//----------------------------------------------------------
int main() {
    demo_operators_basic();
    demo_set_of_points();
    demo_operator_plus();
    demo_functor_as_comparator();
}
