// smart_pointers_demo.cpp
// Kratak "šalabahter" za čas – primeri curenja memorije i pametnih pokazivača.
// Fokus: std::unique_ptr (i kratko shared_ptr/weak_ptr).

#include <iostream>
#include <memory>
#include <vector>
#include <string>

using namespace std;

// =============================================================
// 1) Klasično curenje memorije sa sirovim pokazivačem (T*)
//    - ilustracija problema koji smart pointeri rešavaju
// =============================================================
void demo_raw_leak() {
    cout << "\n[demo_raw_leak]\n";

    int* p = (int*)std::malloc(1000);   // alociramo blok
    if (!p) return;

    // ... nešto radimo sa p ...

    p = (int*)std::malloc(1000);        // PREPISUJEMO adresu
    if (!p) return;

    // free(p);                        // prvi blok NIKAD nije oslobođen → curenje

    std::free(p);                       // oslobađamo samo drugi blok
    cout << "Demonstrirano logičko curenje (overwriting pointer).\n";
}

// =============================================================
// 2) RAII sa unique_ptr – nema ručnog delete
// =============================================================
void demo_unique_basic() {
    cout << "\n[demo_unique_basic]\n";

    std::unique_ptr<int> p = std::make_unique<int>(42);
    cout << "Vrednost: " << *p << "\n";
    // Nema delete p; RAII odradi posao.
}

// =============================================================
// 3) Kopiranje NE RADI, pomeranje RADI
// =============================================================
void demo_unique_move() {
    cout << "\n[demo_unique_move]\n";

    auto a = std::make_unique<int>(7);

    // auto b = a;                 // ❌ NE MOŽE: kopiranje zabranjeno
    auto b = std::move(a);         // ✅ prebacivanje vlasništva

    cout << "*b = " << *b << "\n";
    cout << "a je sada " << (a ? "validan" : "nullptr") << "\n";
}

// =============================================================
// 4) unique_ptr i funkcije
//    - funkcija kojoj ne treba vlasništvo vs ona koja ga preuzima
// =============================================================

// funkcija kojoj NE treba vlasništvo – prosleđujemo referencu
void ispisi(const std::unique_ptr<int>& p) {
    cout << "ispisi: " << *p << "\n";
}

// funkcija koja PREUZIMA vlasništvo (po vrednosti)
void uvecaj_i_zadrzi(std::unique_ptr<int> p) {
    (*p)++;
    cout << "uvecaj_i_zadrzi: " << *p << "\n";
    // p umire na kraju funkcije → objekat se briše ovde
}

void demo_unique_functions() {
    cout << "\n[demo_unique_functions]\n";

    auto p = std::make_unique<int>(10);

    ispisi(p);                  // samo čita, nema std::move

    // uvecaj_i_zadrzi(p);      // ❌ ne radi: p se ne može kopirati
    uvecaj_i_zadrzi(std::move(p));  // ✅ prenos vlasništva

    cout << "Posle std::move, p je "
         << (p ? "i dalje vlasnik" : "null (nema vlasništvo)")
         << "\n";
}

// =============================================================
// 5) vector<unique_ptr<T>> – skladište objekata
// =============================================================
struct Tacka {
    double x, y;
    Tacka(double x, double y) : x(x), y(y) {}
    void prikazi() const { cout << "(" << x << "," << y << ")"; }
};

void demo_vector_unique() {
    cout << "\n[demo_vector_unique]\n";

    std::vector<std::unique_ptr<Tacka>> tacke;

    tacke.push_back(std::make_unique<Tacka>(1.0, 2.0));
    tacke.push_back(std::make_unique<Tacka>(3.5, -0.5));
    tacke.push_back(std::make_unique<Tacka>(0.0, 4.2));

    cout << "Skup tacaka: ";
    for (const auto& t : tacke) {
        t->prikazi();
        cout << " ";
    }
    cout << "\n";

    // prenos vlasništva jedne tačke iz vektora napolje
    auto poslednja = std::move(tacke.back());
    tacke.pop_back()_
