// semantika_premestanja_pregled.cpp
//
// "Šalabahter" za čas o semantici premeštanja (move semantics).
// Nije za studente, već za tebe: kratki, fokusirani primeri
// koje možeš brzo da pokreneš / iseckaš po potrebi.
//
// Tematske celine:
//  1) std::move + unique_ptr – prenos vlasništva (osnovni primer)
//  2) Kopiranje vs. premeštanje za std::string
//  3) Prosleđivanje vektora funkciji po vrednosti: kopija vs. std::move
//  4) Vraćanje vektora iz funkcije: RVO / automatsko pomeranje
//  5) std::move i const objekti – "move" ne pomaže kod const

#include <iostream>
#include <memory>
#include <string>
#include <vector>
#include <utility> // std::move

using std::cout;
using std::endl;

// -----------------------------------------------------------
// 1) std::move + unique_ptr – prenos vlasništva
// -----------------------------------------------------------

void demo_move_unique_ptr() {

    auto p1 = std::make_unique<int>(10);

    // auto p2 = p1; // ❌ neće da kompilira – unique_ptr ne može da se kopira

    auto p2 = std::move(p1); // ✅ prenos vlasništva

    if (!p1)
        cout << "p1 više nije vlasnik (nullptr).\n";

    cout << "p2 sada poseduje vrednost: " << *p2 << endl;

    // Ovde možeš da naglasiš:
    //  - std::move NE briše ništa
    //  - samo označava da je resurs moguće "preuzeti"
}

// -----------------------------------------------------------
// 2) Kopiranje vs. premeštanje za std::string
// -----------------------------------------------------------

void demo_copy_vs_move_string() {

    std::string s1 = "Dugačak neki tekst...";

    // 2.1 Kopiranje
    std::string kopija = s1;  // kopira sadržaj
    cout << "Posle kopije:\n";
    cout << "  s1: " << s1 << "\n";
    cout << "  kopija: " << kopija << "\n";

    // 2.2 Pomeranje
    std::string s2 = std::move(s1); // preuzima interne bafer(e)
    cout << "Posle move:\n";
    cout << "  s1 (prazan ali validan): '" << s1 << "'\n";
    cout << "  s2: '" << s2 << "'\n";

    // Poenta za čas:
    //  - kopija duplira podatke
    //  - move samo "prebaci vlasništvo" nad internom memorijom
}

// -----------------------------------------------------------
// 3) Prosleđivanje vektora funkciji po vrednosti
//    – bez i sa std::move
// -----------------------------------------------------------

void prikazi(std::vector<int> podaci) {
    cout << "  [prikazi] broj elemenata: " << podaci.size() << endl;
}

void demo_move_vector_to_function() {

    std::vector<int> niz{1, 2, 3, 4, 5};

    cout << "Pre poziva funkcije, niz.size() = " << niz.size() << endl;

    // 3.1 Poziv BEZ std::move → pravi se kopija vektora
    cout << "Poziv bez move:\n";
    prikazi(niz);
    cout << "Posle poziva bez move, niz.size() = " << niz.size() << " (i dalje 5)\n";

    // 3.2 Poziv SA std::move → pomeranje (daje vektoru u funkciji sadržaj)
    cout << "Poziv sa std::move:\n";
    prikazi(std::move(niz));
    cout << "Posle std::move, niz.size() = " << niz.size()
         << " (tipično 0 – vektor je 'prazan')\n";

    // Ovde lepo možeš da naglasiš:
    //  - niz je i dalje validan, ali "ispražnjen"
    //  - tipično stanje posle move je size()==0, ali standard ne garantuje tekstualni sadržaj
}

// -----------------------------------------------------------
// 4) Vraćanje vektora iz funkcije – RVO / automatski move
// -----------------------------------------------------------

std::vector<int> napravi_vektor(int n) {
    std::vector<int> v;
    v.reserve(n);
    for (int i = 0; i < n; ++i)
        v.push_back(i);

    // Ovde je dovoljno "return v;" – kompajler će da uradi RVO / move
    return v; // NEMA potrebe za std::move(v)

    // Poenta za čas:
    //  - lokalni objekat koji se vraća **po vrednosti** se gotovo uvek
    //    optimizuje (RVO/NRVO) ili pomera – ručni std::move ovde je višak.
    //
    // "Antipattern" za pokazivanje na tabli (ne mora u kodu):
    //   return std::move(v);
    // – to čak može da onemogući neke optimizacije.
}

// -----------------------------------------------------------
// 5) std::move i const objekti
//    – "move" ne pomaže kod const (poziva se kopija)
// -----------------------------------------------------------

void demo_const_and_move() {
    const std::string cs = "konstanta";

    // std::move(cs) je i dalje const rvalue,
    // pa se ne može "iscediti" – koristi se kopirni konstruktor.
    std::string t = std::move(cs); // praktično kopija, ne "pravi" move

    cout << "cs (const): '" << cs << "'\n";
    cout << "t  (kopija): '" << t << "'\n";

    // Poenta:
    //  - iz const objekta se ne može pomerati sadržaj,
    //    jer bi to promenilo njegovo stanje.
    //  - std::move nad const objektom uglavnom samo bira kopiju,
    //    pa ne donosi dobitak.
}

// -----------------------------------------------------------

int main() {
    demo_move_unique_ptr();          // 1) move i unique_ptr
    demo_copy_vs_move_string();      // 2) string: kopija vs move
    demo_move_vector_to_function();  // 3) vektor u funkciju: kopija vs move
    demo_const_and_move();           // 5) const + std::move

    return 0;
}
