#ifndef LISTA_HPP
#define LISTA_HPP

#include <cstddef>   // std::size_t
#include <memory>    // std::unique_ptr, std::make_unique

namespace dl {

template <typename T>
class lista {
private:
    struct cvor;

public:
    class iterator;

    // --------------------------------------
    // Konstrukcija i destrukcija
    // --------------------------------------
    lista()
        : m_sentinel(), m_poslednji(nullptr), m_broj_elemenata(0)
    {
        m_sentinel.sledeci   = nullptr;
        m_sentinel.prethodni = nullptr;  // ne koristi se za sentinel
    }

    ~lista() {
        ocisti();
    }

    // Kopiranje ne dozvoljavamo (lista sadrži unique_ptr)
    lista(const lista&)            = delete;
    lista& operator=(const lista&) = delete;

    // Move konstruktor i dodela nam nisu potrebni u ovom kursu,
    // pa ih ne definišemo (mada je uvek preporuka u klasama i njih napisati).

    // --------------------------------------
    // Osnovne osobine
    // --------------------------------------
    bool prazna() const {
        return m_broj_elemenata == 0;
    }

    std::size_t velicina() const {
        return m_broj_elemenata;
    }

    void ocisti() {
        // "pustimo" lanac čvorova počev od sentinela
        while (m_sentinel.sledeci) {
            m_sentinel.sledeci =
                std::move(m_sentinel.sledeci->sledeci);
        }
        m_poslednji      = nullptr;
        m_broj_elemenata = 0;
    }

    // --------------------------------------
    // Iterator
    // --------------------------------------
    class iterator {
        cvor*  m_trenutni;
        lista* m_vlasnik;

        iterator(cvor* c, lista* l) : m_trenutni(c), m_vlasnik(l) {}

        friend class lista;

    public:
        iterator() : m_trenutni(nullptr), m_vlasnik(nullptr) {}

        T& operator*() const { return m_trenutni->vrednost; }
        T* operator->() const { return &m_trenutni->vrednost; }

        iterator& operator++() {
            if (m_trenutni) {
                m_trenutni = m_trenutni->sledeci.get();
            }
            return *this;
        }

        iterator operator++(int) {
            iterator tmp(*this);
            ++(*this);
            return tmp;
        }

        iterator& operator--() {
            if (!m_vlasnik) return *this;

            if (m_trenutni == nullptr) {
                // --end() -> poslednji element
                m_trenutni = m_vlasnik->m_poslednji;
            } else {
                m_trenutni = m_trenutni->prethodni;
                if (m_trenutni == &(m_vlasnik->m_sentinel)) {
                    // --begin() je nedefinisano; u korektnom kodu se ne poziva
                    m_trenutni = nullptr;
                }
            }
            return *this;
        }

        iterator operator--(int) {
            iterator tmp(*this);
            --(*this);
            return tmp;
        }

        bool operator==(const iterator& drugi) const {
            return m_trenutni == drugi.m_trenutni;
        }

        bool operator!=(const iterator& drugi) const {
            return !(*this == drugi);
        }
    };

    iterator begin() {
        if (prazna())
            return end();
        return iterator(m_sentinel.sledeci.get(), this);
    }

    iterator end() {
        return iterator(nullptr, this);
    }

    // --------------------------------------
    // Umetanje na početak i kraj
    // --------------------------------------
    void push_front(const T& vrednost) {
        auto novi = std::make_unique<cvor>(vrednost);
        // novi ide odmah iza sentinela
        novi->prethodni = &m_sentinel;
        novi->sledeci   = std::move(m_sentinel.sledeci);

        if (novi->sledeci) {
            novi->sledeci->prethodni = novi.get();
        } else {
            // lista je bila prazna, novi je i poslednji
            m_poslednji = novi.get();
        }

        m_sentinel.sledeci = std::move(novi);
        ++m_broj_elemenata;
    }

    void push_back(const T& vrednost) {
        auto novi = std::make_unique<cvor>(vrednost);
        novi->sledeci = nullptr;

        if (prazna()) {
            // isto kao push_front na praznoj listi
            novi->prethodni      = &m_sentinel;
            m_sentinel.sledeci   = std::move(novi);
            m_poslednji          = m_sentinel.sledeci.get();
        } else {
            novi->prethodni      = m_poslednji;
            m_poslednji->sledeci = std::move(novi);
            m_poslednji          = m_poslednji->sledeci.get();
        }

        ++m_broj_elemenata;
    }

    // --------------------------------------
    // Umetanje i brisanje na poziciji
    // (insert ubacuje PRE pozicije pozicija)
    // --------------------------------------
    iterator insert(iterator pozicija, const T& vrednost) {
        if (pozicija == end()) {
            push_back(vrednost);
            return iterator(m_poslednji, this);
        }

        cvor* tekuci    = pozicija.m_trenutni;
        cvor* prethodni = tekuci->prethodni; // može biti &m_sentinel

        auto novi = std::make_unique<cvor>(vrednost);
        novi->prethodni = prethodni;

        std::unique_ptr<cvor>* veza;
        if (prethodni == &m_sentinel) {
            veza = &m_sentinel.sledeci;
        } else {
            veza = &prethodni->sledeci;
        }

        novi->sledeci = std::move(*veza);   // stari "sledeći" ide iza novog
        if (novi->sledeci) {
            novi->sledeci->prethodni = novi.get();
        } else {
            // umetnuli smo na kraj
            m_poslednji = novi.get();
        }

        *veza = std::move(novi);
        ++m_broj_elemenata;

        return iterator(veza->get(), this);
    }

    iterator erase(iterator pozicija) {
        cvor* tekuci = pozicija.m_trenutni;
        if (!tekuci) return end();

        cvor* prethodni = tekuci->prethodni; // može biti &m_sentinel
        cvor* sledeci   = tekuci->sledeci.get();

        std::unique_ptr<cvor>* veza;
        if (prethodni == &m_sentinel) {
            veza = &m_sentinel.sledeci;
        } else {
            veza = &prethodni->sledeci;
        }

        std::unique_ptr<cvor> za_brisanje = std::move(*veza);
        *veza = std::move(za_brisanje->sledeci);

        if (*veza) {
            (*veza)->prethodni = prethodni;
        } else {
            // obrisali smo poslednji element
            if (prethodni == &m_sentinel) {
                m_poslednji = nullptr;
            } else {
                m_poslednji = prethodni;
            }
        }

        --m_broj_elemenata;

        return iterator(sledeci, this);
    }

private:
    struct cvor {
        T vrednost;
        std::unique_ptr<cvor> sledeci;
        cvor* prethodni;

        cvor() : vrednost(), sledeci(nullptr), prethodni(nullptr) {}

        explicit cvor(const T& v)
            : vrednost(v), sledeci(nullptr), prethodni(nullptr) {}
    };

    cvor        m_sentinel;        // "lažni" čvor na početku liste
    cvor*       m_poslednji;       // poslednji pravi čvor (ili nullptr)
    std::size_t m_broj_elemenata;  // broj pravih elemenata
};

} // namespace dl

#endif // LISTA_HPP
