#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "liste.h"

/* Pročita jedan ceo broj iz niske, koristeći strtol. */
static int procitaj_int(const char **s, int *vrednost)
{
    char *kraj;
    long val = strtol(*s, &kraj, 10);
    if (kraj == *s) {
        return 0;  /* nije bilo broja */
    }
    *vrednost = (int)val;
    *s = kraj;
    return 1;
}

/* Učita listu u formatu [1, 2, 3] sa stdin u zadatu povezanu listu. */
static status ucitaj_listu(cvor *sentinel)
{
    char linija[1024];

    if (fgets(linija, sizeof(linija), stdin) == NULL)
        return NEMA_ELEMENATA;  /* ili tretirati kao OK-prazno */

    const char *p = linija;

    /* lokalni pokazivač na kraj liste – na početku je to sentinel */
    cvor *kraj = sentinel;

    /* Preskoči beline i očekuj '[' */
    while (isspace((unsigned char)*p)) p++;
    if (*p != '[') {
        fprintf(stderr, "Očekivan je znak '['.\n");
        return NEMA_ELEMENATA;
    }
    p++;  /* preko '[' */

    /* Prazna lista '[]'? */
    while (isspace((unsigned char)*p)) p++;
    if (*p == ']') {
        return OK;
    }

    /* Inače čitamo brojeve odvojene zarezima. */
    for (;;) {
        int x;
        while (isspace((unsigned char)*p)) p++;

        if (!procitaj_int(&p, &x)) {
            fprintf(stderr, "Očekivan je ceo broj.\n");
            return NEMA_ELEMENATA;
        }

        if (dodaj_na_kraj_sa_krajem(sentinel, &kraj, x) != OK) {
            fprintf(stderr, "Nedovoljno memorije pri dodavanju elementa.\n");
            return NEDOVOLJNO_MEMORIJE;
        }

        while (isspace((unsigned char)*p)) p++;

        if (*p == ',') {
            p++;  /* preko zareza, ide sledeći broj */
            continue;
        } else if (*p == ']') {
            break;  /* kraj liste */
        } else {
            fprintf(stderr, "Očekivan je ',' ili ']'.\n");
            return NEMA_ELEMENATA;
        }
    }

    return OK;
}


int main(void)
{
    cvor *l1 = inicijalizuj_listu();
    if (l1 == NULL) {
        fprintf(stderr, "Neuspešna alokacija sentinel-čvora.\n");
        return 1;
    }

    printf("Unesite listu u formatu [1, 2, 3]:\n");
    if (ucitaj_listu(l1) != OK) {
        fprintf(stderr, "Greška pri učitavanju liste.\n");
        unisti_listu(&l1);
        return 1;
    }

    printf("Učitana lista: ");
    ispisi_elemente_liste(l1);

    /* Primer: dodavanje na početak i na kraj */
    dodaj_na_pocetak(l1, -1);
    dodaj_na_kraj(l1, 999);

    printf("Nakon dodavanja -1 na početak i 999 na kraj:\n");
    ispisi_elemente_liste(l1);

    /* Primer: brisanje prvog elementa */
    int vrednost;
    if (obrisi_sa_pocetka(l1, &vrednost) == OK) {
        printf("Obrisan prvi element: %d\n", vrednost);
        printf("Lista nakon brisanja sa početka:\n");
        ispisi_elemente_liste(l1);
    }

    /* Primer: brisanje prvog elementa sa zadatom vrednošću */
    int target = 999;
    if (obrisi_prvi_sa_vrednoscu(l1, target) == OK) {
        printf("Obrisan prvi element sa vrednošću %d.\n", target);
        printf("Lista nakon brisanja:\n");
        ispisi_elemente_liste(l1);
    } else {
        printf("Nema elementa sa vrednošću %d.\n", target);
    }

    /* Primer: pravljenje sortirane liste i ubacivanje sortiranjem */
    cvor *sortirana = inicijalizuj_listu();
    if (sortirana == NULL) {
        fprintf(stderr, "Neuspešna alokacija druge liste.\n");
        unisti_listu(&l1);
        return 1;
    }

    ubaci_sortirano(sortirana, 10);
    ubaci_sortirano(sortirana, 5);
    ubaci_sortirano(sortirana, 20);
    ubaci_sortirano(sortirana, 15);

    printf("Sortirana lista (ubacivanje sortiranjem):\n");
    ispisi_elemente_liste(sortirana);

    /* Primer: učešljavanje (merge) dve sortirane liste */
    cvor *sortirana2 = inicijalizuj_listu();
    if (sortirana2 == NULL) {
        fprintf(stderr, "Neuspešna alokacija treće liste.\n");
        unisti_listu(&l1);
        unisti_listu(&sortirana);
        return 1;
    }
    ubaci_sortirano(sortirana2, 7);
    ubaci_sortirano(sortirana2, 8);
    ubaci_sortirano(sortirana2, 30);

    printf("Druga sortirana lista:\n");
    ispisi_elemente_liste(sortirana2);

    printf("Učešljavanje druge u prvu (merge):\n");
    ucesljaj(sortirana, sortirana2);

    printf("Rezultat u prvoj listi:\n");
    ispisi_elemente_liste(sortirana);
    printf("Druga lista posle učešljavanja (trebalo bi da je prazna):\n");
    ispisi_elemente_liste(sortirana2);

    /* Čišćenje */
    unisti_listu(&l1);
    unisti_listu(&sortirana);
    unisti_listu(&sortirana2);

    return 0;
}
